/*
 * Decompiled with CFR 0.152.
 */
package ai.djl.util;

import ai.djl.util.ClassLoaderUtils;
import ai.djl.util.Utils;
import ai.djl.util.cuda.CudaUtils;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Properties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class Platform {
    private static final Logger logger = LoggerFactory.getLogger(Platform.class);
    private String version;
    private String apiVersion;
    private String osPrefix;
    private String osArch;
    private String flavor;
    private String cudaArch;
    private String[] libraries;
    private boolean placeholder;

    private Platform() {
    }

    public static Platform detectPlatform(String engine, String overrideVersion) {
        Platform platform = Platform.fromSystem(engine);
        platform.version = overrideVersion;
        return platform;
    }

    public static Platform detectPlatform(String engine) {
        Enumeration<URL> urls;
        String nativeProp = "native/lib/" + engine + ".properties";
        try {
            urls = ClassLoaderUtils.getContextClassLoader().getResources(nativeProp);
        }
        catch (IOException e) {
            throw new AssertionError("Failed to list property files.", e);
        }
        Platform systemPlatform = Platform.fromSystem(engine);
        ArrayList<Platform> availablePlatforms = new ArrayList<Platform>();
        while (urls.hasMoreElements()) {
            URL url = urls.nextElement();
            Platform platform = Platform.fromUrl(url);
            platform.apiVersion = systemPlatform.apiVersion;
            if (platform.isPlaceholder()) {
                availablePlatforms.add(platform);
                continue;
            }
            if (platform.matches(systemPlatform)) {
                logger.info("Found matching platform from: {}", (Object)url);
                availablePlatforms.add(platform);
                continue;
            }
            logger.info("Ignore mismatching platform from: {}", (Object)url);
        }
        if (availablePlatforms.isEmpty()) {
            if (systemPlatform.version == null) {
                throw new AssertionError((Object)("No " + engine + " version found in property file."));
            }
            if (systemPlatform.apiVersion == null) {
                throw new AssertionError((Object)("No " + engine + " djl_version found in property file."));
            }
            return systemPlatform;
        }
        if (availablePlatforms.size() == 1) {
            Platform ret = (Platform)availablePlatforms.get(0);
            if (ret.isPlaceholder()) {
                logger.info("Found placeholder platform from: {}", (Object)ret);
            }
            return ret;
        }
        availablePlatforms.sort((o1, o2) -> {
            if (o1.isPlaceholder()) {
                return 1;
            }
            if (o2.isPlaceholder()) {
                return -1;
            }
            int ret = o2.getFlavor().compareTo(o1.getFlavor());
            if (ret == 0) {
                return o2.getVersion().compareTo(o1.getVersion());
            }
            return ret;
        });
        return (Platform)availablePlatforms.get(0);
    }

    static Platform fromUrl(URL url) {
        Platform platform = Platform.fromSystem();
        try (InputStream conf = url.openStream();){
            Properties prop = new Properties();
            prop.load(conf);
            platform.version = prop.getProperty("version");
            if (platform.version == null) {
                throw new IllegalArgumentException("version key is required in <engine>.properties file.");
            }
            platform.placeholder = prop.getProperty("placeholder") != null;
            String flavor = prop.getProperty("flavor");
            if (flavor != null) {
                platform.flavor = flavor;
            }
            String flavorPrefixedClassifier = prop.getProperty("classifier", "");
            String libraryList = prop.getProperty("libraries", "");
            platform.libraries = libraryList.isEmpty() ? Utils.EMPTY_ARRAY : libraryList.split(",");
            if (!flavorPrefixedClassifier.isEmpty()) {
                String[] tokens = flavorPrefixedClassifier.split("-");
                if (flavor != null) {
                    platform.osPrefix = tokens[0];
                    platform.osArch = tokens[1];
                } else {
                    platform.flavor = tokens[0];
                    platform.osPrefix = tokens[1];
                    platform.osArch = tokens[2];
                }
            }
        }
        catch (IOException e) {
            throw new IllegalStateException("Failed to read property file: " + url, e);
        }
        return platform;
    }

    public static Platform fromSystem(String engine) {
        Platform platform;
        block8: {
            String engineProp = engine + "-engine.properties";
            String versionKey = engine + "_version";
            platform = Platform.fromSystem();
            platform.placeholder = true;
            try {
                URL url = ClassLoaderUtils.getResource(engineProp);
                if (url == null) break block8;
                try (InputStream is = url.openStream();){
                    Properties prop = new Properties();
                    prop.load(is);
                    platform.version = prop.getProperty(versionKey);
                    platform.apiVersion = prop.getProperty("djl_version");
                }
            }
            catch (IOException e) {
                throw new AssertionError("Failed to read property file: " + engineProp, e);
            }
        }
        return platform;
    }

    static Platform fromSystem() {
        Platform platform = new Platform();
        String osName = System.getProperty("os.name");
        if (osName.startsWith("Win")) {
            platform.osPrefix = "win";
        } else if (osName.startsWith("Mac")) {
            platform.osPrefix = "osx";
        } else if (osName.startsWith("Linux")) {
            platform.osPrefix = "linux";
        } else {
            throw new AssertionError((Object)("Unsupported platform: " + osName));
        }
        platform.osArch = System.getProperty("os.arch");
        if ("amd64".equals(platform.osArch)) {
            platform.osArch = "x86_64";
        }
        if (CudaUtils.getGpuCount() > 0) {
            platform.flavor = "cu" + CudaUtils.getCudaVersionString();
            platform.cudaArch = CudaUtils.getComputeCapability(0);
        } else {
            platform.flavor = "cpu";
        }
        return platform;
    }

    public String getVersion() {
        return this.version;
    }

    public String getApiVersion() {
        return this.apiVersion;
    }

    public String getOsPrefix() {
        return this.osPrefix;
    }

    public String getOsArch() {
        return this.osArch;
    }

    public String getFlavor() {
        return this.flavor;
    }

    public String getClassifier() {
        return this.osPrefix + '-' + this.osArch;
    }

    public String getCudaArch() {
        return this.cudaArch;
    }

    public String[] getLibraries() {
        return this.libraries;
    }

    public boolean isPlaceholder() {
        return this.placeholder;
    }

    public boolean matches(Platform system) {
        if (!this.osPrefix.equals(system.osPrefix) || !this.osArch.equals(system.osArch)) {
            return false;
        }
        if (this.flavor.startsWith("cpu") || "mkl".equals(this.flavor)) {
            return true;
        }
        if (system.flavor.startsWith("cu") && Integer.parseInt(this.flavor.substring(2, 5)) <= Integer.parseInt(system.flavor.substring(2, 5))) {
            return true;
        }
        logger.warn("The bundled library: {}} doesn't match system: {}", (Object)this, (Object)system);
        return false;
    }

    public String toString() {
        return this.flavor + '-' + this.getClassifier() + ':' + this.version;
    }
}

