/*******************************************************************************
NAME                            TRANSVERSE MERCATOR

PURPOSE:	Transforms input Easting and Northing to longitude and
		latitude for the Transverse Mercator projection.  The
		Easting and Northing must be in meters.  The longitude
		and latitude values will be returned in radians.

PROGRAMMER              DATE		REASON
----------              ----		------
D. Steinwand, EROS      Nov, 1991	
T. Mittan		Mar, 1993
S. Nelson		Feb, 1995	Divided tminv.c into two files,
					one for UTM (utminv.c) and one
					for TM (tminv.c).  This was a
					necessary change to run inverse
					projection conversions for both
					UTM and TM in the same process.

ALGORITHM REFERENCES

1.  Snyder, John P., "Map Projections--A Working Manual", U.S. Geological
    Survey Professional Paper 1395 (Supersedes USGS Bulletin 1532), United
    State Government Printing Office, Washington D.C., 1987.

2.  Snyder, John P. and Voxland, Philip M., "An Album of Map Projections",
    U.S. Geological Survey Professional Paper 1453 , United State Government
    Printing Office, Washington D.C., 1989.
*******************************************************************************/
#include <stdio.h>
#include "cproj.h"


/* Variables common to all subroutines in this code file
  -----------------------------------------------------*/
static double r_major;          /* major axis                           */
static double r_minor;          /* minor axis                           */
static double scale_factor;     /* scale factor                         */
static double lon_center;       /* Center longitude (projection center) */
static double lat_origin;       /* center latitude                      */
static double e0,e1,e2,e3;      /* eccentricity constants               */
static double es,esp;         /* eccentricity constants               */
static double ml0;              /* small value m                        */
static double false_northing;   /* y offset in meters                   */
static double false_easting; 	/* x offset in meters			*/
static long ind;		/* sphere flag value			*/


/* Initialize the Transverse Mercator (TM) projection
  -------------------------------------------------------------*/
int tminvint(
double r_maj,			/* major axis			*/
double r_min,			/* minor axis			*/
double scale_fact,		/* scale factor			*/
double center_lon,		/* center longitude		*/
double center_lat,		/* center latitude		*/
double false_east,		/* x offset in meters		*/
double false_north)		/* y offset in meters		*/
{
double temp;			/* temporary variable		*/

/* Place parameters in static storage for common use
  -------------------------------------------------*/
r_major = r_maj;
r_minor = r_min;
scale_factor = scale_fact;
lon_center = center_lon;
lat_origin = center_lat;
false_northing = false_north;
false_easting = false_east;

temp = r_minor / r_major;
es   = 1.0 - SQUARE(temp);
e0   = e0fn(es);
e1   = e1fn(es);
e2   = e2fn(es);
e3   = e3fn(es);
ml0  = r_major * mlfn(e0, e1, e2, e3, lat_origin);
esp  = es / (1.0 - es);

if (es < .00001)
   ind = 1;

/* Report parameters to the user
  -----------------------------*/
ptitle("TRANSVERSE MERCATOR (TM)"); 
radius2(r_major, r_minor);
genrpt(scale_factor,"Scale Factor at C. Meridian:    ");
cenlonmer(lon_center);
origin(lat_origin);
offsetp(false_easting,false_northing);
return(OK);
}

/* Transverse Mercator inverse equations--mapping x,y to lat,long 
   Note:  The algorithm for UTM is exactly the same as TM and therefore
          if a change is implemented, also make the change to UTMINV.c
  --------------------------------------------------------------*/
int tminv(
double x,		/* (I) X projection coordinate 			*/
double y,		/* (I) Y projection coordinate 			*/
double *lon,		/* (O) Longitude 				*/
double *lat)		/* (O) Latitude 				*/
{
double con,phi;		/* temporary angles				*/
double delta_phi;	/* difference between longitudes		*/
long i;			/* counter variable				*/
double sin_phi, cos_phi, tan_phi;	/* sin cos and tangent values	*/
double c, cs, t, ts, n, r, d, ds;	/* temporary variables		*/
double f, h, g, temp;			/* temporary variables		*/
long max_iter = 6;			/* maximun number of iterations	*/

/* fortran code for spherical form 
--------------------------------*/
if (ind != 0)
   {
   f = exp(x/(r_major * scale_factor));
   g = .5 * (f - 1/f);
   temp = lat_origin + y/(r_major * scale_factor);
   h = cos(temp);
   con = sqrt((1.0 - h * h)/(1.0 + g * g));
   *lat = asinz(con);
   if (temp < 0)
     *lat = -*lat;
   if ((g == 0) && (h == 0))
     {
     *lon = lon_center;
     return(OK);
     }
   else
     {
     *lon = adjust_lon(atan2(g,h) + lon_center);
     return(OK);
     }
   }

/* Inverse equations
  -----------------*/
x = x - false_easting;
y = y - false_northing;

con = (ml0 + y / scale_factor) / r_major;
phi = con;
for (i=0;;i++)
   {
   delta_phi = ((con + e1 * sin(2.0*phi) - e2 * sin(4.0*phi) + e3 * sin(6.0*phi))
               / e0) - phi;
/*
   delta_phi = ((con + e1 * sin(2.0*phi) - e2 * sin(4.0*phi)) / e0) - phi;
*/
   phi += delta_phi;
   if (fabs(delta_phi) <= EPSLN) break;
   if (i >= max_iter) 
      { 
      p_error("Latitude failed to converge","TM-INVERSE"); 
      return(95);
      }
   }
if (fabs(phi) < HALF_PI)
   {
   tsincos(phi, &sin_phi, &cos_phi);
   tan_phi = tan(phi);
   c    = esp * SQUARE(cos_phi);
   cs   = SQUARE(c);
   t    = SQUARE(tan_phi);
   ts   = SQUARE(t);
   con  = 1.0 - es * SQUARE(sin_phi); 
   n    = r_major / sqrt(con);
   r    = n * (1.0 - es) / con;
   d    = x / (n * scale_factor);
   ds   = SQUARE(d);
   *lat = phi - (n * tan_phi * ds / r) * (0.5 - ds / 24.0 * (5.0 + 3.0 * t + 
          10.0 * c - 4.0 * cs - 9.0 * esp - ds / 30.0 * (61.0 + 90.0 * t +
          298.0 * c + 45.0 * ts - 252.0 * esp - 3.0 * cs)));
   *lon = adjust_lon(lon_center + (d * (1.0 - ds / 6.0 * (1.0 + 2.0 * t +
          c - ds / 20.0 * (5.0 - 2.0 * c + 28.0 * t - 3.0 * cs + 8.0 * esp +
          24.0 * ts))) / cos_phi));
   }
else
   {
   *lat = HALF_PI * sign(y);
   *lon = lon_center;
   }
return(OK);
}
