/*-
 * SPDX-License-Identifier: BSD-2-Clause
 *
 * Copyright 2006-2008 John Birrell <jb@FreeBSD.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Statically Defined Tracing (SDT) definitions.
 *
 */

#ifndef _SYS_SDT_H
#define	_SYS_SDT_H

#ifndef _KERNEL

#define	_DTRACE_VERSION	1

#define	DTRACE_PROBE(prov, name) {				\
	extern void __dtrace_##prov##___##name(void);		\
	__dtrace_##prov##___##name();				\
}

#define	DTRACE_PROBE1(prov, name, arg1) {			\
	extern void __dtrace_##prov##___##name(unsigned long);	\
	__dtrace_##prov##___##name((unsigned long)arg1);	\
}

#define	DTRACE_PROBE2(prov, name, arg1, arg2) {			\
	extern void __dtrace_##prov##___##name(unsigned long,	\
	    unsigned long);					\
	__dtrace_##prov##___##name((unsigned long)arg1,		\
	    (unsigned long)arg2);				\
}

#define	DTRACE_PROBE3(prov, name, arg1, arg2, arg3) {		\
	extern void __dtrace_##prov##___##name(unsigned long,	\
	    unsigned long, unsigned long);			\
	__dtrace_##prov##___##name((unsigned long)arg1,		\
	    (unsigned long)arg2, (unsigned long)arg3);		\
}

#define	DTRACE_PROBE4(prov, name, arg1, arg2, arg3, arg4) {	\
	extern void __dtrace_##prov##___##name(unsigned long,	\
	    unsigned long, unsigned long, unsigned long);	\
	__dtrace_##prov##___##name((unsigned long)arg1,		\
	    (unsigned long)arg2, (unsigned long)arg3,		\
	    (unsigned long)arg4);				\
}

#define	DTRACE_PROBE5(prov, name, arg1, arg2, arg3, arg4, arg5) {	\
	extern void __dtrace_##prov##___##name(unsigned long,		\
	    unsigned long, unsigned long, unsigned long, unsigned long);\
	__dtrace_##prov##___##name((unsigned long)arg1,			\
	    (unsigned long)arg2, (unsigned long)arg3,			\
	    (unsigned long)arg4, (unsigned long)arg5);			\
}

#else /* _KERNEL */

#include <sys/linker_set.h>
#include <machine/sdt_machdep.h>

extern volatile bool sdt_probes_enabled;

#ifndef KDTRACE_HOOKS

#define __sdt_used	__unused

#define SDT_PROVIDER_DEFINE(prov)
#define SDT_PROVIDER_DECLARE(prov)
#define SDT_PROBE_DEFINE(prov, mod, func, name)
#define SDT_PROBE_DECLARE(prov, mod, func, name)
#define SDT_PROBES_ENABLED()	0
#define SDT_PROBE(prov, mod, func, name, arg0, arg1, arg2, arg3, arg4)
#define SDT_PROBE_ARGTYPE(prov, mod, func, name, num, type, xtype)

#define	SDT_PROBE_DEFINE0(prov, mod, func, name)
#define	SDT_PROBE_DEFINE1(prov, mod, func, name, arg0)
#define	SDT_PROBE_DEFINE2(prov, mod, func, name, arg0, arg1)
#define	SDT_PROBE_DEFINE3(prov, mod, func, name, arg0, arg1, arg2)
#define	SDT_PROBE_DEFINE4(prov, mod, func, name, arg0, arg1, arg2, arg3)
#define	SDT_PROBE_DEFINE5(prov, mod, func, name, arg0, arg1, arg2, arg3, arg4)
#define	SDT_PROBE_DEFINE6(prov, mod, func, name, arg0, arg1, arg2,      \
    arg3, arg4, arg5)

#define	SDT_PROBE0(prov, mod, func, name)
#define	SDT_PROBE1(prov, mod, func, name, arg0)
#define	SDT_PROBE2(prov, mod, func, name, arg0, arg1)
#define	SDT_PROBE3(prov, mod, func, name, arg0, arg1, arg2)
#define	SDT_PROBE4(prov, mod, func, name, arg0, arg1, arg2, arg3)
#define	SDT_PROBE5(prov, mod, func, name, arg0, arg1, arg2, arg3, arg4)
#define	SDT_PROBE6(prov, mod, func, name, arg0, arg1, arg2, arg3, arg4, arg5)

#define	MIB_SDT_PROBE1(...)
#define	MIB_SDT_PROBE2(...)

#define	SDT_PROBE_DEFINE0_XLATE(prov, mod, func, name)
#define	SDT_PROBE_DEFINE1_XLATE(prov, mod, func, name, arg0, xarg0)
#define	SDT_PROBE_DEFINE2_XLATE(prov, mod, func, name, arg0, xarg0,     \
    arg1, xarg1)
#define	SDT_PROBE_DEFINE3_XLATE(prov, mod, func, name, arg0, xarg0,     \
    arg1, xarg1, arg2, xarg2)
#define SDT_PROBE_DEFINE4_XLATE(prov, mod, func, name, arg0, xarg0,     \
    arg1, xarg1, arg2, xarg2, arg3, xarg3)
#define	SDT_PROBE_DEFINE5_XLATE(prov, mod, func, name, arg0, xarg0,     \
    arg1, xarg1, arg2, xarg2, arg3, xarg3, arg4, xarg4)
#define	SDT_PROBE_DEFINE6_XLATE(prov, mod, func, name, arg0, xarg0,     \
    arg1,  xarg1, arg2, xarg2, arg3, xarg3, arg4, xarg4, arg5, xarg5)

#define	DTRACE_PROBE(name)
#define	DTRACE_PROBE1(name, type0, arg0)
#define	DTRACE_PROBE2(name, type0, arg0, type1, arg1)
#define	DTRACE_PROBE3(name, type0, arg0, type1, arg1, type2, arg2)
#define	DTRACE_PROBE4(name, type0, arg0, type1, arg1, type2, arg2, type3, arg3)
#define	DTRACE_PROBE5(name, type0, arg0, type1, arg1, type2, arg2, type3, arg3,\
    type4, arg4)

#else

void sdt_probe(uint32_t, uintptr_t, uintptr_t, uintptr_t, uintptr_t,
    uintptr_t);
void sdt_probe6(uint32_t, uintptr_t, uintptr_t, uintptr_t, uintptr_t,
    uintptr_t, uintptr_t);

#define	_SDT_TRACEPOINT_SET		sdt_tracepoint_set
#define	_SDT_TRACEPOINT_SECTION		"set_sdt_tracepoint_set"

bool sdt_tracepoint_valid(uintptr_t patchpoint, uintptr_t target);
void sdt_tracepoint_patch(uintptr_t patchpoint, uintptr_t target);
void sdt_tracepoint_restore(uintptr_t patchpoint);

#define __sdt_used

SET_DECLARE(sdt_providers_set, struct sdt_provider);
SET_DECLARE(sdt_probes_set, struct sdt_probe);
SET_DECLARE(sdt_argtypes_set, struct sdt_argtype);

#define	_SDT_PROBE_NAME(prov, mod, func, name)				\
	sdt_##prov##_##mod##_##func##_##name
#define	_SDT_PROVIDER_NAME(prov)					\
	sdt_provider_##prov

#define SDT_PROVIDER_DEFINE(_prov)					\
	struct sdt_provider _SDT_PROVIDER_NAME(_prov) = {		\
		.name = #_prov,						\
	};								\
	DATA_SET(sdt_providers_set, _SDT_PROVIDER_NAME(_prov))

#define SDT_PROVIDER_DECLARE(prov)					\
	extern struct sdt_provider _SDT_PROVIDER_NAME(prov)

#define SDT_PROBE_DEFINE(_prov, _mod, _func, _name)			\
	struct sdt_probe _SDT_PROBE_NAME(_prov, _mod, _func, _name) = {	\
		.version = sizeof(struct sdt_probe),			\
		.prov = &_SDT_PROVIDER_NAME(_prov),			\
		.mod = #_mod,						\
		.func = #_func,						\
		.name = #_name,						\
	};								\
	DATA_SET(sdt_probes_set, _SDT_PROBE_NAME(_prov, _mod, _func, _name))

#define SDT_PROBE_DECLARE(prov, mod, func, name)			\
	extern struct sdt_probe _SDT_PROBE_NAME(prov, mod, func, name)

#define	SDT_PROBES_ENABLED()	__predict_false(sdt_probes_enabled)

#ifdef _ILP32
#define	_SDT_ASM_WORD			".long"
#else
#define	_SDT_ASM_WORD			".quad"
#endif

#ifndef _SDT_ASM_PROBE_CONSTRAINT
#define	_SDT_ASM_PROBE_CONSTRAINT	"i"
#endif
#ifndef	_SDT_ASM_PROBE_OPERAND
#if !defined(__clang__) && __GNUC_PREREQ__(15, 0)
#define	_SDT_ASM_PROBE_OPERAND		"cc"
#else
#define	_SDT_ASM_PROBE_OPERAND		"c"
#endif
#endif

/*
 * The asm below generates records corresponding to the structure's layout, so
 * the two must be kept in sync.
 */
struct sdt_tracepoint {
	struct sdt_probe *probe;
	uintptr_t	patchpoint;
	uintptr_t	target;
	STAILQ_ENTRY(sdt_tracepoint) tracepoint_entry;
};

#define __SDT_PROBE(prov, mod, func, name, uniq, f, ...) do {		\
	__WEAK(__CONCAT(__start_set_, _SDT_TRACEPOINT_SET));		\
	__WEAK(__CONCAT(__stop_set_, _SDT_TRACEPOINT_SET));		\
	asm goto(							\
	    "0:\n"							\
	    _SDT_ASM_PATCH_INSTR "\n"					\
	    ".pushsection " _SDT_TRACEPOINT_SECTION ", \"aw\"\n"	\
	    _SDT_ASM_WORD " %" _SDT_ASM_PROBE_OPERAND "0\n"		\
	    _SDT_ASM_WORD " 0b\n"					\
	    _SDT_ASM_WORD " %l1\n"					\
	    _SDT_ASM_WORD " 0\n"					\
	    ".popsection\n"						\
	    :								\
	    : _SDT_ASM_PROBE_CONSTRAINT (&_SDT_PROBE_NAME(prov, mod,	\
	    func, name))						\
	    :								\
	    : __sdt_probe##uniq);					\
	if (0) {							\
__sdt_probe##uniq:;							\
		f(_SDT_PROBE_NAME(prov, mod, func, name).id, __VA_ARGS__); \
	}								\
} while (0)
#define _SDT_PROBE(prov, mod, func, name, uniq, f, ...)			\
	__SDT_PROBE(prov, mod, func, name, uniq, f, __VA_ARGS__)
#define SDT_PROBE(prov, mod, func, name, arg0, arg1, arg2, arg3, arg4)	\
	_SDT_PROBE(prov, mod, func, name, __COUNTER__, sdt_probe,	\
	    (uintptr_t)arg0, (uintptr_t)arg1, (uintptr_t)arg2,		\
	    (uintptr_t)arg3, (uintptr_t)arg4)

#define SDT_PROBE_ARGTYPE(_prov, _mod, _func, _name, _num, _type, _xtype) \
	static struct sdt_argtype					\
	    sdta_##_prov##_##_mod##_##_func##_##_name##_num[1] = {	\
		[0] = {							\
		    .ndx = _num,					\
		    .type = _type,					\
		    .xtype = _xtype,					\
		    .probe = &_SDT_PROBE_NAME(_prov, _mod, _func, _name), \
		},							\
	};								\
	DATA_SET(sdt_argtypes_set,					\
	    sdta_##_prov##_##_mod##_##_func##_##_name##_num);

#define	SDT_PROBE_DEFINE0(prov, mod, func, name)			\
	SDT_PROBE_DEFINE(prov, mod, func, name)

#define	SDT_PROBE_DEFINE1(prov, mod, func, name, arg0)			\
	SDT_PROBE_DEFINE(prov, mod, func, name);			\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 0, arg0, NULL)

#define	SDT_PROBE_DEFINE2(prov, mod, func, name, arg0, arg1)		\
	SDT_PROBE_DEFINE(prov, mod, func, name);			\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 0, arg0, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 1, arg1, NULL)

#define	SDT_PROBE_DEFINE3(prov, mod, func, name, arg0, arg1, arg2)\
	SDT_PROBE_DEFINE(prov, mod, func, name);			\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 0, arg0, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 1, arg1, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 2, arg2, NULL)

#define	SDT_PROBE_DEFINE4(prov, mod, func, name, arg0, arg1, arg2, arg3) \
	SDT_PROBE_DEFINE(prov, mod, func, name);			\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 0, arg0, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 1, arg1, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 2, arg2, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 3, arg3, NULL)

#define	SDT_PROBE_DEFINE5(prov, mod, func, name, arg0, arg1, arg2, arg3, arg4) \
	SDT_PROBE_DEFINE(prov, mod, func, name);			\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 0, arg0, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 1, arg1, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 2, arg2, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 3, arg3, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 4, arg4, NULL)

#define	SDT_PROBE_DEFINE6(prov, mod, func, name, arg0, arg1, arg2, arg3,\
    arg4, arg5) \
	SDT_PROBE_DEFINE(prov, mod, func, name);			\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 0, arg0, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 1, arg1, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 2, arg2, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 3, arg3, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 4, arg4, NULL);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 5, arg5, NULL)

#define	SDT_PROBE_DEFINE0_XLATE(prov, mod, func, name)		\
	SDT_PROBE_DEFINE(prov, mod, func, name)

#define	SDT_PROBE_DEFINE1_XLATE(prov, mod, func, name, arg0, xarg0) \
	SDT_PROBE_DEFINE(prov, mod, func, name);			\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 0, arg0, xarg0)

#define	SDT_PROBE_DEFINE2_XLATE(prov, mod, func, name, arg0, xarg0, \
    arg1,  xarg1)							\
	SDT_PROBE_DEFINE(prov, mod, func, name);			\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 0, arg0, xarg0);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 1, arg1, xarg1)

#define	SDT_PROBE_DEFINE3_XLATE(prov, mod, func, name, arg0, xarg0, \
    arg1, xarg1, arg2, xarg2)						\
	SDT_PROBE_DEFINE(prov, mod, func, name);			\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 0, arg0, xarg0);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 1, arg1, xarg1);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 2, arg2, xarg2)

#define	SDT_PROBE_DEFINE4_XLATE(prov, mod, func, name, arg0, xarg0, \
    arg1, xarg1, arg2, xarg2, arg3, xarg3)				\
	SDT_PROBE_DEFINE(prov, mod, func, name);			\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 0, arg0, xarg0);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 1, arg1, xarg1);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 2, arg2, xarg2);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 3, arg3, xarg3)

#define	SDT_PROBE_DEFINE5_XLATE(prov, mod, func, name, arg0, xarg0, \
    arg1, xarg1, arg2, xarg2, arg3, xarg3, arg4, xarg4)			\
	SDT_PROBE_DEFINE(prov, mod, func, name);			\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 0, arg0, xarg0);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 1, arg1, xarg1);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 2, arg2, xarg2);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 3, arg3, xarg3);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 4, arg4, xarg4)

#define	SDT_PROBE_DEFINE6_XLATE(prov, mod, func, name, arg0, xarg0, \
    arg1, xarg1, arg2, xarg2, arg3, xarg3, arg4, xarg4, arg5, xarg5)	\
	SDT_PROBE_DEFINE(prov, mod, func, name);			\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 0, arg0, xarg0);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 1, arg1, xarg1);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 2, arg2, xarg2);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 3, arg3, xarg3);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 4, arg4, xarg4);	\
	SDT_PROBE_ARGTYPE(prov, mod, func, name, 5, arg5, xarg5)

#define	SDT_PROBE0(prov, mod, func, name)				\
	SDT_PROBE(prov, mod, func, name, 0, 0, 0, 0, 0)
#define	SDT_PROBE1(prov, mod, func, name, arg0)				\
	SDT_PROBE(prov, mod, func, name, arg0, 0, 0, 0, 0)
#define	SDT_PROBE2(prov, mod, func, name, arg0, arg1)			\
	SDT_PROBE(prov, mod, func, name, arg0, arg1, 0, 0, 0)
#define	SDT_PROBE3(prov, mod, func, name, arg0, arg1, arg2)		\
	SDT_PROBE(prov, mod, func, name, arg0, arg1, arg2,  0, 0)
#define	SDT_PROBE4(prov, mod, func, name, arg0, arg1, arg2, arg3)	\
	SDT_PROBE(prov, mod, func, name, arg0, arg1, arg2, arg3, 0)
#define	SDT_PROBE5(prov, mod, func, name, arg0, arg1, arg2, arg3, arg4) \
	SDT_PROBE(prov, mod, func, name, arg0, arg1, arg2, arg3, arg4)
#define	SDT_PROBE6(prov, mod, func, name, arg0, arg1, arg2, arg3, arg4, arg5) \
	_SDT_PROBE(prov, mod, func, name, __COUNTER__, sdt_probe6,	\
	    (uintptr_t)arg0, (uintptr_t)arg1, (uintptr_t)arg2,		\
	    (uintptr_t)arg3, (uintptr_t)arg4, (uintptr_t)arg5)

#ifdef KDTRACE_MIB_SDT
#define	MIB_SDT_PROBE1(...)	SDT_PROBE1(mib, __VA_ARGS__)
#define	MIB_SDT_PROBE2(...)	SDT_PROBE2(mib, __VA_ARGS__)
#else
#define	MIB_SDT_PROBE1(...)
#define	MIB_SDT_PROBE2(...)
#endif

#define	DTRACE_PROBE_IMPL_START(name, arg0, arg1, arg2, arg3, arg4)	do { \
	static SDT_PROBE_DEFINE(sdt, , , name);				     \
	SDT_PROBE(sdt, , , name, arg0, arg1, arg2, arg3, arg4);
#define DTRACE_PROBE_IMPL_END	} while (0)

#define DTRACE_PROBE(name)						\
	DTRACE_PROBE_IMPL_START(name, 0, 0, 0, 0, 0)			\
	DTRACE_PROBE_IMPL_END

#define DTRACE_PROBE1(name, type0, arg0)				\
	DTRACE_PROBE_IMPL_START(name, arg0, 0, 0, 0, 0) 		\
	SDT_PROBE_ARGTYPE(sdt, , , name, 0, #type0, NULL);		\
	DTRACE_PROBE_IMPL_END

#define DTRACE_PROBE2(name, type0, arg0, type1, arg1)			\
	DTRACE_PROBE_IMPL_START(name, arg0, arg1, 0, 0, 0) 		\
	SDT_PROBE_ARGTYPE(sdt, , , name, 0, #type0, NULL);		\
	SDT_PROBE_ARGTYPE(sdt, , , name, 1, #type1, NULL);		\
	DTRACE_PROBE_IMPL_END

#define DTRACE_PROBE3(name, type0, arg0, type1, arg1, type2, arg2)	\
	DTRACE_PROBE_IMPL_START(name, arg0, arg1, arg2, 0, 0)	 	\
	SDT_PROBE_ARGTYPE(sdt, , , name, 0, #type0, NULL);		\
	SDT_PROBE_ARGTYPE(sdt, , , name, 1, #type1, NULL);		\
	SDT_PROBE_ARGTYPE(sdt, , , name, 2, #type2, NULL);		\
	DTRACE_PROBE_IMPL_END

#define DTRACE_PROBE4(name, type0, arg0, type1, arg1, type2, arg2, type3, arg3)	\
	DTRACE_PROBE_IMPL_START(name, arg0, arg1, arg2, arg3, 0) 	\
	SDT_PROBE_ARGTYPE(sdt, , , name, 0, #type0, NULL);		\
	SDT_PROBE_ARGTYPE(sdt, , , name, 1, #type1, NULL);		\
	SDT_PROBE_ARGTYPE(sdt, , , name, 2, #type2, NULL);		\
	SDT_PROBE_ARGTYPE(sdt, , , name, 3, #type3, NULL);		\
	DTRACE_PROBE_IMPL_END

#define DTRACE_PROBE5(name, type0, arg0, type1, arg1, type2, arg2, type3, arg3,	\
    type4, arg4)								\
	DTRACE_PROBE_IMPL_START(name, arg0, arg1, arg2, arg3, arg4) 	\
	SDT_PROBE_ARGTYPE(sdt, , , name, 0, #type0, NULL);		\
	SDT_PROBE_ARGTYPE(sdt, , , name, 1, #type1, NULL);		\
	SDT_PROBE_ARGTYPE(sdt, , , name, 2, #type2, NULL);		\
	SDT_PROBE_ARGTYPE(sdt, , , name, 3, #type3, NULL);		\
	SDT_PROBE_ARGTYPE(sdt, , , name, 4, #type4, NULL);		\
	DTRACE_PROBE_IMPL_END

#endif /* KDTRACE_HOOKS */

/*
 * This type definition must match that of dtrace_probe. It is defined this
 * way to avoid having to rely on CDDL code.
 */
typedef	void (*sdt_probe_func_t)(uint32_t, uintptr_t arg0, uintptr_t arg1,
    uintptr_t arg2, uintptr_t arg3, uintptr_t arg4, uintptr_t arg5);

/*
 * The 'sdt' provider will set it to dtrace_probe when it loads.
 */
extern sdt_probe_func_t		sdt_probe_func;

struct sdt_probe;
struct sdt_provider;
struct linker_file;

struct sdt_argtype {
	int		ndx;		/* Argument index. */
	const char	*type;		/* Argument type string. */
	const char	*xtype;		/* Translated argument type. */
	TAILQ_ENTRY(sdt_argtype)
			argtype_entry;	/* Argument type list entry. */
	struct sdt_probe *probe;	/* Ptr to the probe structure. */
};

struct sdt_probe {
	int		version;	/* Set to sizeof(struct sdt_probe). */
	struct sdt_provider *prov;	/* Ptr to the provider structure. */
	TAILQ_ENTRY(sdt_probe)
			probe_entry;	/* SDT probe list entry. */
	TAILQ_HEAD(, sdt_argtype) argtype_list;
	STAILQ_HEAD(, sdt_tracepoint) tracepoint_list;
	const char	*mod;
	const char	*func;
	const char	*name;
	uint32_t	id;		/* DTrace probe ID. */
	int		n_args;		/* Number of arguments. */
	struct linker_file *sdtp_lf;	/* Module in which we're defined. */
};

struct sdt_provider {
	char *name;			/* Provider name. */
	TAILQ_ENTRY(sdt_provider)
			prov_entry;	/* SDT provider list entry. */
	uintptr_t	id;		/* DTrace provider ID. */
	int		sdt_refs;	/* Number of module references. */
};

void sdt_probe_stub(uint32_t, uintptr_t, uintptr_t, uintptr_t, uintptr_t,
    uintptr_t, uintptr_t);

SDT_PROVIDER_DECLARE(sdt);

#endif /* _KERNEL */

#endif /* _SYS_SDT_H */
